/*
 * TurtleGraphics.java
 *
 * Created on February 22, 2003, 2:35 PM
 *
 * Copyright 2003 Danny Brewer
 * Anyone may run this code.
 * If you wish to modify or distribute this code, then
 *  you are granted a license to do so only under the terms
 *  of the Gnu Lesser General Public License.
 * See:  http://www.gnu.org/licenses/lgpl.html
 */

package nom.DannyBrewer.graphics;

/**
 * <p>
 * This class is an interface for drawing Turtle Graphics.
 * </p>
 *
 * <p>
 * The main concept in Turtle Graphics is that there is a turtle positioned\
 *  on the paper.
 * The turtle has a pen.  The pen may be up or down.
 * You can move the turtle forward (or backward) by some number of units.
 * When turtle motion occurs, if the pen is down, a line is drawn.
 * You can tell the turtle to turn by a certian angle.
 * </p>
 *
 * <p>
 * For example, if you move by 5 units, turn 90 degrees, and then do this
 *  three more times, you will have drawn a square and the turtle will be
 *  in its original starting position, and pointed in the same direction
 *  it was originally pointed.
 * </p>
 *
 * <p>Initial conditions:
 *    (Implementations of this interface should strive to have
 *    these initial conditions when the turtle is instantiated.)</p>
 * <p>* Turtle at center of paper
 *     (if that makes sense for the particular implementation of this interface,
 *     otherwise turtle should be at upper-left if that makes sense).</p>
 * <p>* Turtle pointed at zero degrees.
 *    (3 O'Clock, East, or Right.)</p>
 * <p>* Turtle angles specified in degrees.</p>
 *
 * @author  danny brewer
 */
public interface TurtleGraphics {
    
    
    /*
     * <p>
     * Point the turtle in a certian direction specified by angle.
     * Add angle to the current angle.
     * </p>
     */
    public void turn( double angle );
    /*
     * <p>
     * Point the turtle to an absolute angle.
     * </p>
     */
    public void turnTo( double angle );
    /*
     * <p>
     * This returns what direction angle the turtle is currently pointed.
     * </p>
     */
    public double getTurtleDirection();
    
    
    /*
     * <p>
     * If you want to measure angles in Radians
     *  then FULL_CIRCLE_RADIANS is the number of units
     *  which make up a full circle.
     * There are 2*PI radians in a circle.
     * </p>
     * <p>Call setFullCircleAngle( FULL_CIRCLE_RADIANS );</p>
     */
    public static final double FULL_CIRCLE_RADIANS = 2.0 * Math.PI;	// 2*pi radians in a circle
    /*
     * <p>
     * If you want to measure angles in Degrees
     *  then FULL_CIRCLE_DEGREES is the number of units
     *  which make up a full circle.
     * There are 360 degrees in a circle.
     * </p>
     * <p>Call setFullCircleAngle( FULL_CIRCLE_DEGREES );</p>
     */
    public static final double FULL_CIRCLE_DEGREES = 360;			// 360 degress in a circle
    /*
     * <p>
     * If you want to measure angles in Gradians
     *  then FULL_CIRCLE_GRADIANS is the number of units
     *  which make up a full circle.
     * There are 400 gradians in a circle.
     * </p>
     * <p>Call setFullCircleAngle( FULL_CIRCLE_GRADIANS );</p>
     */
    public static final double FULL_CIRCLE_GRADIANS = 400;			// 400 gradians in a circle
    /*
     * <p>
     * Change what units the angles are measured in.
     * </p>
     * <p>
     * If you wanted to change to measuring angles in radians,
     *  you could simply call...
     * </p>
     * <p>
     *		setFullCircleAngle( FULL_CIRCLE_RADIANS )
     * </p>
     * <p>
     *  to change angular measurements to radians.
     * </p>
     * <p>
     * To convert back to using degrees for angular measurements,
     *  simply call either...
     * </p>
     * <p>
     *		setFullCircleAngle( FULL_CIRCLE_DEGREES )
     * </p>
     * <p>
     *		setFullCircleAngle( 360 )
     * </p>
     * <p>
     * You can measure angles in any arbitrary units you wish.
     * For instance, to measure angles as fractions of a circle,
     *  just specify 1.0 as the full circle measurement.
     * </p>
     * <p>
     *		setFullCircleAngle( 1.0 )
     * </p>
     * <p>
     * A right angle would now be measured as 0.25.
     * A full turnaround would be measured as 0.5.
     * A full circle would be measured as 1.0.
     * </p>
     */
    public void setFullCircleAngle( double fullCircleAngle );
    public double getFullCircleAngle();
    
    /*
     * <p>
     * A utility function.
     * No matter what units the turtle is using for angular measurement,
     *  this function converts an angle into radians.
     * </p>
     */
    public double convertAngleToRadians( double angle );
    
    
    
    /*
     * <p>Is the turtle's pen Down?</p>
     */
    public boolean isPenDown();
    /*
     * <p>Is the turtle's pen Up?</p>
     */
    public boolean isPenUp();
    /*
     * <p>Put the turtle's pen down.</p>
     */
    public void penDown() ;
    /*
     * <p>Put the turtle's pen up.</p>
     */
    public void penUp();
    /*
     * <p>Put the turtle's pen up or down according to the parameter.</p>
     */
    public void penDown( boolean down );
    
    
    /*
     * <p>
     * Move the turtle some distance in its present direction.
     * Whether a line is drawn depends on whether the pen is down.
     * </p>
     */
    public void moveForward( int distance );
    // Move the turtle to an absolute location.
    // The turtle remains pointed in the same direction it was.
    public void moveTo( int x, int y );
    // Move the turtle a relative amount.
    // The turtle remains pointed in the same direction it was.
    public void moveBy( int detaX, int deltaY );
    
    
    // Move the turtle some distance in its present direction.
    // The pen will be put down and a line will be drawn.
    public void lineForward( int distance );
    // Draw line in an absolute location.
    // The turtle remains pointed in the same direction it was.
    public void lineTo( int x, int y );
    // Draw line by a relative amount.
    // The turtle remains pointed in the same direction it was.
    public void lineBy( int detaX, int deltaY );
    
    
    // Move the turtle some distance in its present direction.
    // The pen will be put up and no line will be drawn.
    public void jumpForward( int distance );
    // Move the turtle to an absolute location.
    // The turtle remains pointed in the same direction it was.
    public void jumpTo( int x, int y );
    // Move the turtle a relative amount.
    // The turtle remains pointed in the same direction it was.
    public void jumpBy( int detaX, int deltaY );
    
    
    public int getTurtleXPosition();
    public int getTurtleYPosition();
}
